#!/bin/sh
#
#/*******************************************************************
# * This file is part of the Emulex Linux Device Driver for         *
# * Fibre Channel Host Bus Adapters.                                *
# * Copyright (C) 2003-2005 Emulex.  All rights reserved.           *
# * EMULEX and SLI are trademarks of Emulex.                        *
# * www.emulex.com                                                  *
# *                                                                 *
# * This program is free software; you can redistribute it and/or   *
# * modify it under the terms of version 2 of the GNU General       *
# * Public License as published by the Free Software Foundation.    *
# * This program is distributed in the hope that it will be useful. *
# * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND          *
# * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,  *
# * FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE      *
# * DISCLAIMED, EXCEPT TO THE EXTENT THAT SUCH DISCLAIMERS ARE HELD *
# * TO BE LEGALLY INVALID.  See the GNU General Public License for  *
# * more details, a copy of which can be found in the file COPYING  *
# * included with this package.                                     *
# *******************************************************************/
#
# scripts/lun_change_state Rev : 1.1
# This script is provided by Emulex to use with its 8.X linux device 
# driver for Light Pulse Fibre Channel adapters.
#
# This script changes the state of an existing lun (scsi_device) from
# offline to running.  It is intended to run prior to running lun_scan.
# 
# This script runs on a list of specified lpfc HBAs (hosts) or on all
# lpfc HBAs.  When scanning an HBA, all discovered targets, and all 
# luns will be probed.
#
# USAGE: The script is invoked with at least 1 argument. The arguments
# specify either the SCSI Host numbers corresponding to the specific
# lpfc HBA that are to be scanned, or the keyword "all" which indicates
# that all lpfc HBAs are to be scanned.
#
usage()
{
	echo ""
	echo "Usage: lun_change_state [ <#> [<#>] | all ]"
	echo "  where "
	echo "    <#> : is a scsi host number of a specific lpfc HBA that is "
	echo "          providing access to the luns.  More than 1 host number "
	echo "          can be specified. "
	echo "    all : indicates that all lpfc HBAs are to be scanned."
	echo ""
	echo "  Example:"
	echo "    lun_change_state all  : Enables all offline luns on all lpfc HBAs"
	echo "    lun_change_state 2    : Enables all offline luns on the lpfc HBAs with scsi host number 2"
	echo "    lun_change_state 2 4  : Enables all offline luns on the lpfc HBAs with scsi host number 2 and 4"
	echo ""
	echo "  Warning: Scanning an adapter while performing a tape backup should"
	echo "    be avoided."
	echo ""
}

# Validate argument list
if [ $# -eq 0 -o "$1" = "--help" -o "$1" = "-h" ] ; then
	usage
	exit 1
fi


# get list of lpfc HBAs to scan
hosts=$*;

all_hosts=`ls -1 -d /sys/bus/pci/drivers/lpfc/*/host* | sed -e "s/.*host//"`

# If all option is used get all the lpfc host numbers.
if [ "$hosts" == "all" ] ; then
    hosts="$all_hosts"
fi 

cd /sys/class/scsi_device
for host in $hosts ; do
	echo "working on host $host"
	if [ ! -e /sys/bus/pci/drivers/lpfc/*/host$host ] ; then 
		echo ""
		echo "Error: Cannot find an lpfc HBA instance with scsi host number : $host"
		echo "... Aborting lun_change_state."
		echo ""
		exit 2
	fi

	for lun in `ls -d $host*` ; do
		cd $lun/device

		# RHEL 4 distributions have a scsi_device state sysfs attribute.
		# SLES 9 distributions have a scsi_device online sysfs attribute.
		if [ -e state ] ; then
			lun_state=`cat state`
			if [ "$lun_state" == "offline" ] ; then
				echo "Found lun $lun in state $lun_state, Modifying to Running"
				echo running > state
			fi
		elif [ -e online ] ; then
			lun_online=`cat online`
			if [ "$lun_online" == "0" ] ; then
				echo "Found lun $lun offline, issuing rescan"
				echo 1 > online
				echo 1 > rescan 
			fi
		else
			echo "Sysfs attributes state or online not present for scsi_device $lun"
		fi
		cd ../..
	done
done

echo ""
